#!/bin/bash

################################################################
# LPDDR4パラメータの破損チェック
#   chkLPDDR4Param.sh シリアルNo
#   【引数】
#       シリアルNo：ログファイルに付加するシリアルNo
#
#   【戻り値】
#       LPDDR4パラメータ正常：0
#       LPDDR4パラメータ異常：1
#       診断エラー、治具不正：255
#   【履歴】
#       Ver.1.2.0：OP_BTS-39568：LPDDR4追加パラメータ更新対応
#       Ver.1.2.1：OP_BTS-39987(IT6_2.2MR4)/40117(IT6_2.2MR3)：BOOT DL 1回目NG問題対応.
#
################################################################

### 定数定義 ###
Version="Ver.1.2.1"
fileDevMtd2="/dev/mtd2"

### 事前処理 ###
pathRecoveryJig=$(cd $(dirname $0); pwd)  # 復旧治具のパスを取得
SerialNo=${1}

if [ ! -d "${pathRecoveryJig}/Log/" ]; then
    # ログの保存先を作成
    mkdir -p "${pathRecoveryJig}/Log/"
fi
fileLog="${pathRecoveryJig}/Log/RecoveryLog_LPDDR_${SerialNo}_"$(date +"%Y%m%d_%H%M%S")".log" # ログファイル名

echo "[LPDDR4 Param] Start" >> ${fileLog}

if [ 0 -eq $(ls ${pathRecoveryJig}/LP4_PARAM_*.bin| wc -w) ]; then
	# 期待値データファイルが見つからない
    echo "[LPDDR4 Param] No expected data found." >> ${fileLog}

    ls "${pathRecoveryJig}/" >> ${fileLog}
    echo "[LPDDR4 Param] $0 $Version" >> ${fileLog}
    exit 255
fi

# SPI-Flashデバイス(/dev/mtd2)の存在を確認
if [ ! -c ${fileDevMtd2} ]; then
    echo "[LPDDR4 Param] File not found:${fileDevMtd2}" >> ${fileLog}
    ls -l "/dev" >> ${fileLog}
    echo "[LPDDR4 Param] $0 $Version" >> ${fileLog}
    exit 1
fi


################################################################
#
# LPDDR4パラメータの抽出
#   【引数】なし
#   【戻値】なし
#
################################################################
function Extract_Param()
{
  # SPI-FlashからLPDDR4パラメータの抽出(16進数2桁ずつ)
  {
      dataLPDDR4Param=`dd if=${fileDevMtd2} bs=1024 skip=518 count=2 | od -A n -t x1 -v`
  } >> ${fileLog} 2>&1
  echo "[LPDDR4 Param] LPDDR4 Param:${dataLPDDR4Param}" >> ${fileLog}
}

################################################################
#
# LPDDR4パラメータの更新
#   【引数】なし
#   【戻値】
#       更新完了：0
#       更新ファイルなし：1
#
################################################################
UpdateFile=""
function Update_Param()
{
  if [ ! -e ${UpdateFile} ]; then
    echo "[LPDDR4 Param]"${UpdateFile}" not found." >> ${fileLog}
    return 1
  fi

  # SPI-FLASHのクリアは、4KB単位だが
  # LPDDRパラメータ領域は、オフセットが0x81800であるため
  # 0x81000～0x817ffの2KB分を書き戻し用にダンプする
  dd if=${fileDevMtd2} of="${pathRecoveryJig}/lpddr_backup.bin" bs=1024 skip=516 count=2

  # 書き戻しダンプと更新ファイルを結合.
  cat "${pathRecoveryJig}/lpddr_backup.bin" ${UpdateFile} > "${pathRecoveryJig}/lpddr_restore.bin"

  # LPDDRパラメータ領域のWriteプロテクト解除
  echo 0 > /sys/devices/platform/quartz-sb/8e8273000.spi0/write_protect_lpddr4param

  # SPI-FLASHの書き込み領域を消去（4KB）
  ${pathRecoveryJig}/ioctl_if /dev/mtd2 3 0x81000 0x1000

  if [ $? -ne 0 ]; then
    echo "ioctl_if error"
    # LPDDRパラメータ領域のWriteプロテクトを戻す
    echo 1 > /sys/devices/platform/quartz-sb/8e8273000.spi0/write_protect_lpddr4param
    return 1
  fi

  # SPI-FlashのLPDDR4パラメータを更新する
  {
    dd if="${pathRecoveryJig}/lpddr_restore.bin" of=${fileDevMtd2} bs=1024 seek=516 count=4 conv=notrunc
  } >> ${fileLog} 2>&1

  # LPDDRパラメータ領域のWriteプロテクトを戻す
# echo 1 > /sys/devices/platform/quartz-sb/8e8273000.spi0/write_protect_lpddr4param

  if [ $? -ne 0 ]; then
    echo "LPDDR4_PARAM write fail" >> ${fileLog}
  else
    echo "LPDDR4_PARAM write success" >> ${fileLog}
  fi

  return 0
}

################################################################
#
# LPDDR4パラメータのチェック
#   【引数】なし
#   【戻値】
#       パラメータが一致しない：0
#       パラメータが一致した：1
#       パラメータが一致したが更新：2
#
################################################################
function Check_Param()
{
  ret=0
  # 治具内の期待値データを列挙し、抽出したLPDDR4パラメータと合致するものを探す。
  for fileExpectedData in ${pathRecoveryJig}/LP4_PARAM_*.bin
      do
        {
          # バイナリ比較
          {
            for _byte in ${dataLPDDR4Param}
              do
                printf "\x${_byte}"
              done
          } | cmp -l ${fileExpectedData}
          result=$?
        } > "/dev/null" 2>&1

        # データの結果を検証
        if [ ${result} -eq 0 ]; then
          # 抽出したLPDDR4パラメータと合致した
          echo "[LPDDR4 Param] OK" >> ${fileLog}

          # 期待値データのフルパスからメーカー情報を生成
          MakerInfoArray=(`basename ${fileExpectedData} | sed -E "s/LP4_PARAM_(.+)\.bin/\1/g" | sed -e "s/_/ /g"`)
          ret=1

          # Samsungの場合のみ新パラメータへ置き換え
          if [ ${MakerInfoArray[0]} = "Samsung" ];  then
            # メモリサイズ(4G/6G/8Gが対象)
            case ${MakerInfoArray[1]} in
              #Samsung 4G s 2100/2400
              "4G")
                case ${MakerInfoArray[2]} in
                  "s")
                  case ${MakerInfoArray[3]} in
                    "2100")
                      echo "Samsung 4G s 2100 Param need update!"
                      echo "Samsung 4G s 2100 Param need update!" >> ${fileLog}
                      UpdateFile="${pathRecoveryJig}/LP4_PARAM_Samsung_r222_4G_s_2100.bin"
                      ret=2
                      ;;
                    "2400")
                      echo "Samsung 4G s 2400 Param need update!"
                      echo "Samsung 4G s 2400 Param need update!" >> ${fileLog}
                      UpdateFile="${pathRecoveryJig}/LP4_PARAM_Samsung_r232_4G_s_2400.bin"
                      ret=2
                      ;;
                  esac;;
                esac;;

              #Samsung 6G 2400
              "6G")
                case ${MakerInfoArray[2]} in
                  "2400")
                    echo "Samsung 6G 2400 Param need update!"
                    echo "Samsung 6G 2400 Param need update!" >> ${fileLog}
                    UpdateFile="${pathRecoveryJig}/LP4_PARAM_Samsung_r232_6G_2400.bin"
                    ret=2
                    ;;
                esac;;

              #Samsung 8G 2100/2400
              "8G")
                case ${MakerInfoArray[2]} in
                  "2100")
                    echo "Samsung 8G 2100 Param need update!"
                    echo "Samsung 8G 2100 Param need update!" >> ${fileLog}
                    UpdateFile="${pathRecoveryJig}/LP4_PARAM_Samsung_r222_8G_2100.bin"
                    ret=2
                    ;;
                  "2400")
                    echo "Samsung 8G 2400 Param need update!"
                    echo "Samsung 8G 2400 Param need update!" >> ${fileLog}
                    UpdateFile="${pathRecoveryJig}/LP4_PARAM_Samsung_r232_8G_2400.bin"
                    ret=2
                    ;;
                esac;;

              #Samsung other
              *)
                echo "Samsung No Need to update!" >> ${fileLog}
                ;;
            esac
          fi
 
         # Micronの場合のみ新パラメータへ置き換え
         if [ ${MakerInfoArray[0]} = "Micron" ];  then
            # メモリサイズ(3G/4G/6G/8Gが対象)
            case ${MakerInfoArray[1]} in
              #Micron 3G s 2400
              "3G")
                case ${MakerInfoArray[2]} in
                  "s")
                  case ${MakerInfoArray[3]} in
                    "2400")
                      echo "Micron 3G s 2400 Param need update!"
                      echo "Micron 3G s 2400 Param need update!" >> ${fileLog}
                      UpdateFile="${pathRecoveryJig}/LP4_PARAM_Micron_r232_3G_s_2400.bin"
                      ret=2
                      ;;
                  esac;;
                esac;;

              #Micron 4G s 2100/2400
              "4G")
                case ${MakerInfoArray[2]} in
                  "s")
                  case ${MakerInfoArray[3]} in
                    "2100")
                      echo "Micron 4G s 2100 Param need update!"
                      echo "Micron 4G s 2100 Param need update!" >> ${fileLog}
                      UpdateFile="${pathRecoveryJig}/LP4_PARAM_Micron_r222_4G_s_2100.bin"
                      ret=2
                      ;;
                    "2400")
                      echo "Micron 4G s 2400 Param need update!"
                      echo "Micron 4G s 2400 Param need update!" >> ${fileLog}
                      UpdateFile="${pathRecoveryJig}/LP4_PARAM_Micron_r242_4G_s_2400.bin"
                      ret=2
                      ;;
                  esac;;
                esac;;

              #Micron 6G 2400
              "6G")
                case ${MakerInfoArray[2]} in
                  "2400")
                    echo "Micron 6G 2400 Param need update!"
                    echo "Micron 6G 2400 Param need update!" >> ${fileLog}
                    UpdateFile="${pathRecoveryJig}/LP4_PARAM_Micron_r232_6G_2400.bin"
                    ret=2
                    ;;
                esac;;

              #Micron 8G s 2100/2400
              "8G")
                case ${MakerInfoArray[2]} in
                  "s")
                  case ${MakerInfoArray[3]} in
                    "2100")
                      echo "Micron 8G s 2100 Param need update!"
                      echo "Micron 8G s 2100 Param need update!" >> ${fileLog}
                      UpdateFile="${pathRecoveryJig}/LP4_PARAM_Micron_r222_8G_s_2100.bin"
                      ret=2
                      ;;
                    "2400")
                      echo "Micron 8G s 2400 Param need update!"
                      echo "Micron 8G s 2400 Param need update!" >> ${fileLog}
                      UpdateFile="${pathRecoveryJig}/LP4_PARAM_Micron_r232_8G_s_2400.bin"
                      ret=2
                      ;;
                  esac;;
                esac;;

              #Micron other
              *)
                echo "Micron No Need to update!" >> ${fileLog}
                ;;
            esac
          fi

          if [ ${ret} -eq 1 ]; then
              # 更新なしで一致するデータがあった
              # メーカー情報を表示する（標準出力とログファイル両方に）
              # 期待値データのフルパスからメーカー情報を生成（ここではパラメータのrevは削除）
              MakerInfo=`basename ${fileExpectedData} | sed -E "s/LP4_PARAM_(.+)\.bin/\1/g" | sed -e "s/r2[234]2_//g" | sed -e "s/_/ /g"`
              echo "Maker: ${MakerInfo}"
              echo "Maker: ${MakerInfo}" >> ${fileLog}
          fi

          break
        fi
      done
  return ${ret}
}

### 処理 ###
# LPDDR4パラメータの抽出
Extract_Param

# LPDDR4パラメータのチェック
Check_Param
_ExpectedDataMatched=$?

if [ ${_ExpectedDataMatched} -eq 0 ]; then
    # データは一致しなかった
    echo "[LPDDR4 Param] NG(Differ from the expected data)" >> ${fileLog}
    result=1 # 実施結果：1

elif [ ${_ExpectedDataMatched} -eq 2 ]; then
    # パラメータ更新
    Update_Param
    Update_result=$?
    if [ ${Update_result} -eq 1 ]; then
        # 更新ファイルが見つからなかった
        echo "[LPDDR4 Param] NG(Update File not found)"
        echo "[LPDDR4 Param] NG(Update File not found)" >> ${fileLog}
        result=4 # 実施結果：4

    else
        # 再度抽出
        echo "[LPDDR4 Param] Update data check" >> ${fileLog}
        Extract_Param
        # 再チェック
        Check_Param
        _ExpectedDataMatched=$?
        if [ ${_ExpectedDataMatched} -eq 0 ]; then
            # データは一致しなかった
            echo "[LPDDR4 Param] Update NG(Differ from the expected data)" 
            echo "[LPDDR4 Param] Update NG(Differ from the expected data)" >> ${fileLog}
            result=3 # 実施結果：3

        elif [ ${_ExpectedDataMatched} -eq 2 ]; then
            # 更新完了後に再度更新必要パラメータと一致した（更新できていない）
            echo "[LPDDR4 Param] Update NG(Matched with previous data)"
            echo "[LPDDR4 Param] Update NG(Matched with previous data)" >> ${fileLog}
            result=3 # 実施結果：3

        else
            # 更新完了後データ一致
            echo "[LPDDR4 Param] Update OK" >> ${fileLog}
            result=2 # 実施結果：2
        fi
    fi
fi

echo "[LPDDR4 Param] $0 $Version" >> ${fileLog}
exit ${result}
